  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Tue Nov 30 10:17:10 2008
   *  Last modified   :  %modify_time%
   */
  /** @file 
   *  \brief This file includes the RSA ATP tests
   *
   *  \version CRYS_SEP_ATP_RSA_Acceptance.c#1:csrc:1
   *  \author 
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************* Include Files ****************/
#include "CRYS.h"
#include "CRYS_SEP_ATP_Defines.h"
#include "DX_VOS_Mem.h"
#include "CRYS_SEP_ATP_RSA_Acceptance.h"

/************************ Defines *******************************/
/************************ Enums *********************************/
/************************ Typedefs ******************************/
/************************ Global Data ***************************/

extern CRYSError_t      TST_ATP_Error;
/************* Private function prototype ***********************/
/************************ Public Functions ******************************/
CRYSError_t SEPQA_ATP_CheckErrorReturn(CRYSError_t  TST_ErrorRet,
									   char         TST_FuncName[TST_FUNC_STRING_LEN],
									   DxUint8_t   *FailureFunctionNameOut_ptr);

/* global variables could occupied only 1K,   */
/* so all variables are allocated as internal */
/* variables of the functions                 */ 



/***********************************************************************
* Function Name:                                                           
*  SEPQA_RSA_PRIME_EncDec         
* 
* Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* CRT_Mode - mode of the private key
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
*  dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* OutputLocation - indication if input buffer == output buffer or not.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
* This function is qa external application on the SEP, 
* that executes RSA primitive encryption/decryption process
* Algorithm:   
* 1. Allocate memory buffers
* 2. Build the Public Key from External exponent and modulus
* 3. Build the Private Key from external exponent and modulus
* 4. Encrypt a known vector
* 5. Compare the outputted result to the expected one                                                   
* 6. Decrypt the encrypted output
* 7. Compare the outputted result to the original text
* 6. Free allocated buffers
*
**************************************************************************************/                                

CRYSError_t SEPQA_RSA_PRIME_EncDec (DxUint8_t	*PubExponent_ptr,
									DxUint16_t	 PubExponentSize,
									DxUint8_t   *PrivExponent_ptr,
									DxUint16_t   PrivExponentSize,
									DxUint8_t	*Modulus_ptr,
									DxUint16_t	 KeySize,
									DxUint8_t    CRT_Mode,
									DxUint8_t   *P_ptr, 
									DxUint16_t	 PSize,
									DxUint8_t   *Q_ptr,
									DxUint16_t   QSize,
									DxUint8_t   *dP_ptr, 
									DxUint16_t   dPSize,
									DxUint8_t   *dQ_ptr,
									DxUint16_t   dQSize,
									DxUint8_t   *qInv_ptr,
									DxUint16_t   qInvSize,
									DxUint8_t   *Data_ptr,
									DxUint16_t   DataSize,
									DxUint8_t   *ExpectedData_ptr,
									DxUint8_t   *FailureFunctionNameOut)
{
	/*** TST Variables Definitions ****/
	CRYSError_t					TST_TestStatus = 0;
	DxUint16_t					TST_GetModulusSize = 0;
	DxUint8_t					*pDataIn;
	DxUint8_t					*pDataOut;
	CRYS_RSAUserPrivKey_t       *TST_PrivateKey1;
	CRYS_RSAUserPubKey_t        *TST_PublicKey1;
	CRYS_RSAPrimeData_t         *TST_PrimeData1;

	/* 1. Allocate memory buffers */
	/******************************/
	pDataIn			= DX_VOS_MemMalloc(DataSize);
	pDataOut		= DX_VOS_MemMalloc(KeySize);
	TST_PrivateKey1 = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPrivKey_t));
	TST_PublicKey1	= DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPubKey_t));
	TST_PrimeData1	= DX_VOS_MemMalloc(sizeof(CRYS_RSAPrimeData_t));	

	if ((TST_PrivateKey1 == DX_NULL)     ||
		(TST_PublicKey1 == DX_NULL)      ||
		(TST_PrimeData1 == DX_NULL)      ||
		(pDataIn == DX_NULL)             ||
		(pDataOut == DX_NULL))
	{

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
			"DX_VOS_MemMalloc",
			FailureFunctionNameOut);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, Data_ptr, DataSize);

	/*Cleaning the keys to prevent test errors with KG*/ 
	DX_VOS_MemSet(TST_PrivateKey1,0,sizeof(CRYS_RSAUserPrivKey_t));
	DX_VOS_MemSet(TST_PublicKey1,0,sizeof(CRYS_RSAUserPubKey_t));
	DX_VOS_MemSet(TST_PrimeData1,0,sizeof(CRYS_RSAPrimeData_t));


	/* 2. Build the public Key */
	/***************************/
	TST_ATP_Error = CRYS_RSA_Build_PubKey(TST_PublicKey1,
		PubExponent_ptr,
		PubExponentSize,
		Modulus_ptr,
		KeySize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_RSA_Build_PubKey",
		FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* Get the E size */
	/*----------------*/
	TST_ATP_Error = CRYS_RSA_Get_ModSizeFromPubKey(TST_PublicKey1,
		&TST_GetModulusSize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_RSA_Get_ModSizeFromPubKey",
		FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* Check that received E size is correct */
	/*---------------------------------------*/
	TST_TestStatus = TST_GetModulusSize != PubExponentSize ? TESTFAIL : TESTPASS;

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"Compare - TST_GetModulusSize",
		FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 3. Build the Private Key */
	/****************************/ 
	if (CRT_Mode == TST_NO_CRT)
	{
		TST_ATP_Error = CRYS_RSA_Build_PrivKey(TST_PrivateKey1,
			PrivExponent_ptr,
			KeySize,
			PubExponent_ptr,
			PubExponentSize,
			Modulus_ptr,
			KeySize);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
			"CRYS_RSA_Build_PrivKey",
			FailureFunctionNameOut);

		if (TST_TestStatus != TESTPASS) 
			goto End;			
	}
	else
	{
		TST_ATP_Error = CRYS_RSA_Build_PrivKeyCRT(TST_PrivateKey1,
													P_ptr,
													PSize,
													Q_ptr,
													QSize,
													dP_ptr,
													dPSize,
													dQ_ptr,
													dQSize,
													qInv_ptr,
													qInvSize);


		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
			"CRYS_RSA_Build_PrivKeyCRT",
			FailureFunctionNameOut);

		if (TST_TestStatus != TESTPASS) 
			goto End;			
	}

	/* 4. Encrypt the data using the primes */
	/* (Public Key)                         */
	/****************************************/ 
	TST_ATP_Error = CRYS_RSA_PRIM_Encrypt(TST_PublicKey1,
		TST_PrimeData1,
		pDataIn,
		DataSize,
		pDataOut);   



	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_RSA_PRIM_Encrypt",
		FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;			


	/* 5. Compare the Encrypted data to the expected */
	/*************************************************/
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pDataOut,
		ExpectedData_ptr,
		KeySize);                  

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"DX_VOS_MemCmp",
		FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;			

	/* 6. Decrypt the data using the primes */
	/* (Private Key)                        */
	/****************************************/ 
	TST_ATP_Error = CRYS_RSA_PRIM_Decrypt(TST_PrivateKey1,
		TST_PrimeData1,
		pDataOut,
		DataSize,
		pDataOut);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_RSA_PRIM_Decrypt",
		FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;			


	/* 7. Compare the outputted result to the original text */
	/********************************************************/
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pDataOut,
		pDataIn,
		DataSize);  

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"DX_VOS_MemCmp",
		FailureFunctionNameOut);
End:
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);
	DX_VOS_MemFree(TST_PrivateKey1);
	DX_VOS_MemFree(TST_PublicKey1);
	DX_VOS_MemFree(TST_PrimeData1);

	return TST_TestStatus;	

}

/***********************************************************************
* Function Name:                                                           
*  SEPQA_RSA_PRIME_SignVerify         
* 
* Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* CRT_Mode - mode of the private key
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
*  dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* OutputLocation - indication if input buffer == output buffer or not.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
* FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
*
* Outputs:
*  CRYSError_t - Function error return                                     
* Description:                                                                 
*  This function Is a Acceptance test for RSA primitive Sign & Verify                  
*  This function signs and verifies known vectors using RSA primitives
*  Vector/Keys sizes used (in bits) - 2048                                                        
*  This also outputs the results to an output buffer or the same input buffer
* Algorithm:   
* 1. Allocate memory buffers
* 2. Build the Private Key from external exponent and modulus
* 3. Build the Public Key from External exponent and modulus
* 4. sign/verify a known vector
* 5. Compare the outputted result to the expected one
* 6. Free allocated buffers
*
**************************************************************************************/

CRYSError_t SEPQA_RSA_PRIME_SignVerify (DxUint8_t	*PubExponent_ptr,
										DxUint16_t	 PubExponentSize,
										DxUint8_t   *PrivExponent_ptr,
										DxUint8_t	*Modulus_ptr,
										DxUint16_t	 KeySize,
										DxUint8_t    CRT_Mode,
										DxUint8_t   *P_ptr, 
										DxUint16_t	 PSize,
										DxUint8_t   *Q_ptr,
										DxUint16_t   QSize,
										DxUint8_t   *dP_ptr, 
										DxUint16_t   dPSize,
										DxUint8_t   *dQ_ptr,
										DxUint16_t   dQSize,
										DxUint8_t   *qInv_ptr,
										DxUint16_t   qInvSize,
										DxUint8_t   *Data_ptr,
										DxUint16_t   DataSize,
										DxUint8_t   *ExpectedData_ptr,
										DxUint8_t   *FailureFunctionNameOut)
{
	/*** TST Variables Definitions ****/
	CRYSError_t					TST_TestStatus = 0;
	DxUint16_t					TST_GetModulusSize = 0;
	DxUint8_t					*pDataIn;
	DxUint8_t					*pDataOut;
	CRYS_RSAUserPrivKey_t       *TST_PrivateKey1;
	CRYS_RSAUserPubKey_t        *TST_PublicKey1;
	CRYS_RSAPrimeData_t         *TST_PrimeData1;

	/* 1. Allocate memory buffers */
	/******************************/
	pDataIn			= DX_VOS_MemMalloc(DataSize);
	pDataOut		= DX_VOS_MemMalloc(KeySize);
	TST_PrivateKey1 = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPrivKey_t));
	TST_PublicKey1	= DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPubKey_t));
	TST_PrimeData1	= DX_VOS_MemMalloc(sizeof(CRYS_RSAPrimeData_t));	

	if ((TST_PrivateKey1 == DX_NULL)     ||
		(TST_PublicKey1 == DX_NULL)      ||
		(TST_PrimeData1 == DX_NULL)      ||
		(pDataIn == DX_NULL)             ||
		(pDataOut == DX_NULL))
	{

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
													"DX_VOS_MemMalloc",
													FailureFunctionNameOut);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, Data_ptr, DataSize);

	/*Cleaning the keys to prevent test errors with KG*/ 
	DX_VOS_MemSet(TST_PrivateKey1,0,sizeof(CRYS_RSAUserPrivKey_t));
	DX_VOS_MemSet(TST_PublicKey1,0,sizeof(CRYS_RSAUserPubKey_t));
	DX_VOS_MemSet(TST_PrimeData1,0,sizeof(CRYS_RSAPrimeData_t));

	/* 2. Build the public Key */
	/***************************/
	TST_ATP_Error = CRYS_RSA_Build_PubKey(TST_PublicKey1, PubExponent_ptr, 
		PubExponentSize, Modulus_ptr, KeySize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_RSA_Build_PubKey",
												FailureFunctionNameOut);
	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* Get the E size */
	/*----------------*/
	TST_ATP_Error = CRYS_RSA_Get_ModSizeFromPubKey(TST_PublicKey1,	&TST_GetModulusSize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_RSA_Get_ModSizeFromPubKey",
												FailureFunctionNameOut);
	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* Check that received E size is correct */
	/*---------------------------------------*/
	TST_TestStatus = TST_GetModulusSize != PubExponentSize ? TESTFAIL : TESTPASS;

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"Compare - TST_GetModulusSize",
												FailureFunctionNameOut);
	if (TST_TestStatus != TESTPASS) 
		goto End;    

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_RSA_Build_PubKey",
												FailureFunctionNameOut);
	if (TST_TestStatus != TESTPASS) 
		goto End;  


	/* 3. Build the Private Key */
	/****************************/  
	if (CRT_Mode == TST_NO_CRT)
	{
		TST_ATP_Error = CRYS_RSA_Build_PrivKey(TST_PrivateKey1,PrivExponent_ptr,
											   KeySize, PubExponent_ptr, PubExponentSize,                                                   
											   Modulus_ptr, KeySize);                                  

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_RSA_Build_PrivKey",
													FailureFunctionNameOut);
		if (TST_TestStatus != TESTPASS) 
			goto End;  

	}
	else
	{
		TST_ATP_Error = CRYS_RSA_Build_PrivKeyCRT(TST_PrivateKey1, P_ptr,PSize,
												  Q_ptr, QSize, dP_ptr, dPSize, 
												  dQ_ptr, dQSize, qInv_ptr, qInvSize);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_RSA_Build_PrivKeyCRT",
													FailureFunctionNameOut);
		if (TST_TestStatus != TESTPASS) 
			goto End; 
	}

	/* 4. Verify the data using the primes */
	/* (Public Key)                         */
	/****************************************/ 
	TST_ATP_Error = CRYS_RSA_PRIM_Verify(TST_PublicKey1, TST_PrimeData1,
										 pDataIn, KeySize, pDataOut);    

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_RSA_PRIM_Verify",
												FailureFunctionNameOut);
	if (TST_TestStatus != TESTPASS) 
		goto End; 


	/* 5. Compare the Encrypted data to the expected */
	/*************************************************/
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pDataOut ,ExpectedData_ptr, KeySize);                  

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"DX_VOS_MemCmp",
												FailureFunctionNameOut);
	if (TST_TestStatus != TESTPASS) 
		goto End; 

	/* 6. Verify the data using the primes */
	/* (Private Key)                         */
	/****************************************/    
	TST_ATP_Error = CRYS_RSA_PRIM_Sign(TST_PrivateKey1,
									   TST_PrimeData1,
									   pDataOut,
									   KeySize,
									   pDataOut);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_RSA_PRIM_Sign",
												FailureFunctionNameOut);
	if (TST_TestStatus != TESTPASS) 
		goto End; 

	/* 7. Compare the outputted result to the original text */
	/********************************************************/
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pDataOut ,pDataIn,KeySize);  

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"DX_VOS_MemCmp - Sign",
												FailureFunctionNameOut);
	if (TST_TestStatus != TESTPASS) 
		goto End; 

End:
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);
	DX_VOS_MemFree(TST_PrivateKey1);
	DX_VOS_MemFree(TST_PublicKey1);
	DX_VOS_MemFree(TST_PrimeData1);

	return TST_TestStatus;	
}

/***********************************************************************
* Function Name:                                                           
*  SEPQA_RSA_PKCS1v15_EncDec         
* 
* Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* CRT_Mode - mode of the private key
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
*  dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* OutputLocation - indication if input buffer == output buffer or not.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
* Description:
* This function is qa external application on the SEP, that executes
* RSA PKCS1v15 encrypt and decrypt according to PKCS#1 Ver1.5
* Algorithm:   
* 1. Allocate memory buffers
* 2. Prepare keys 
*    a. Generate keys 
*    or
*    a. Build the Private Key from external exponent and modulus
*    b. Build the Public Key from External exponent and modulus
* 3. Encrypt a known vector
* 5. Decrypt the data using the PKCS1v15 
* 6. Compare the outputted result to the input one                  
* 7. Free allocated buffers
*
**************************************************************************************/                                
CRYSError_t SEPQA_RSA_PKCS1v15_EncDec (DxUint8_t	*PubExponent_ptr,
									   DxUint16_t	 PubExponentSize,
									   DxUint8_t   *PrivExponent_ptr,
									   DxUint16_t   PrivExponentSize,
									   DxUint8_t	*Modulus_ptr,
									   DxUint16_t	 KeySize,
									   DxUint16_t   KeyGenType,
									   DxUint8_t    CRT_Mode,
									   DxUint8_t   *P_ptr, 
									   DxUint16_t	 PSize,
									   DxUint8_t   *Q_ptr,
									   DxUint16_t   QSize,
									   DxUint8_t   *dP_ptr, 
									   DxUint16_t   dPSize,
									   DxUint8_t   *dQ_ptr,
									   DxUint16_t   dQSize,
									   DxUint8_t   *qInv_ptr,
									   DxUint16_t   qInvSize,
									   DxUint8_t   *Data_ptr,
									   DxUint16_t   DataSize,
									   DxUint8_t   *FailureFunctionNameOut)
{
	/*** TST Variables Definitions ****/
	DxUint16_t    TST_ActualDecDataSize; 
	DxUint8_t	  *pDataIn;
	DxUint8_t	  *pDataOut;
	CRYSError_t    TST_TestStatus=0;
	CRYS_RSAUserPrivKey_t       *TST_PrivateKey1;
	CRYS_RSAUserPubKey_t        *TST_PublicKey1;
	CRYS_RSAKGData_t            *TST_KeyGenData1;
	CRYS_RSAPrimeData_t         *TST_PrimeData1;
	

	/* 1. Allocate memory buffers */
	/******************************/
	pDataIn = DX_VOS_MemMalloc(DataSize);
	pDataOut = DX_VOS_MemMalloc(KeySize);

	TST_PrivateKey1 = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPrivKey_t));
	TST_PublicKey1 = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPubKey_t));
	TST_KeyGenData1 = DX_VOS_MemMalloc(sizeof(CRYS_RSAKGData_t));
	TST_PrimeData1 = DX_VOS_MemMalloc(sizeof(CRYS_RSAPrimeData_t));	

	if ((TST_PrivateKey1 == DX_NULL)     ||
		(TST_PublicKey1 == DX_NULL)      ||
		(TST_KeyGenData1 == DX_NULL)     ||
		(TST_PrimeData1 == DX_NULL)      ||
		(pDataIn == DX_NULL)             ||
		(pDataOut == DX_NULL))
	{
		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
													"DX_VOS_MemMalloc",
													FailureFunctionNameOut);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, Data_ptr, DataSize);

	TST_ActualDecDataSize = (DxUint16_t)(KeySize*8);

	/*Cleaning the keys to prevent test errors with KG*/ 
	DX_VOS_MemSet(TST_PrivateKey1,0,sizeof(CRYS_RSAUserPrivKey_t));
	DX_VOS_MemSet(TST_PublicKey1,0,sizeof(CRYS_RSAUserPubKey_t));
	DX_VOS_MemSet(TST_KeyGenData1,0,sizeof(CRYS_RSAKGData_t));
	DX_VOS_MemSet(TST_PrimeData1,0,sizeof(CRYS_RSAPrimeData_t));


	/* 2. Prepare keys */
	/*******************/
	if (KeyGenType == TST_INTERNAL_KG)
	{
		if (CRT_Mode == TST_NO_CRT)
		{
			/* Generate The Public and private keys */
			/****************************************/
			TST_ATP_Error = CRYS_RSA_KG_GenerateKeyPair(
				PubExponent_ptr,
				PubExponentSize,
				KeySize*8,
				TST_PrivateKey1,
				TST_PublicKey1,
				TST_KeyGenData1);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_KG_GenerateKeyPair",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                                    
		}
		else
		{
			/* Generate The Public and private CRT keys */
			/********************************************/
			TST_ATP_Error = CRYS_RSA_KG_GenerateKeyPairCRT(
				PubExponent_ptr,		
				PubExponentSize,
				KeySize*8,
				TST_PrivateKey1,
				TST_PublicKey1,
				TST_KeyGenData1);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_KG_GenerateKeyPairCRT",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                                    
		}
	}
	else
	{
		/* Build the public Key */
		/*************************/
		TST_ATP_Error = CRYS_RSA_Build_PubKey(
			TST_PublicKey1,
			PubExponent_ptr,
			PubExponentSize,
			Modulus_ptr,
			KeySize);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
			"CRYS_RSA_Build_PubKey",
			FailureFunctionNameOut);

		if (TST_TestStatus != TESTPASS) 
			goto End;                                                                    

		/* Build the Private Key */
		/*************************/ 
		if (CRT_Mode == TST_NO_CRT)
		{
			TST_ATP_Error = CRYS_RSA_Build_PrivKey(TST_PrivateKey1,
				PrivExponent_ptr,
				PrivExponentSize,
				PubExponent_ptr,
				PubExponentSize,                                                   
				Modulus_ptr,
				KeySize);                                  

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_Build_PrivKey",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                                    
		}
		else
		{
			TST_ATP_Error = CRYS_RSA_Build_PrivKeyCRT(TST_PrivateKey1,
				P_ptr,
				PSize,
				Q_ptr,                                                                                                 
				QSize,
				dP_ptr,                                                                                                
				dPSize, 
				dQ_ptr,
				dQSize,
				qInv_ptr,
				qInvSize);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_Build_PrivKeyCRT",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                                    
		}
	}/*End of TST_KeyGenType*/

	/* 3. Encrypt the data using the PKCS1v15 */
	/* (Public Key)                           */
	/******************************************/   
	TST_ATP_Error = CRYS_RSA_PKCS1v15_Encrypt(TST_PublicKey1,
		TST_PrimeData1,
		pDataIn,
		DataSize,
		pDataOut);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_RSA_PKCS1v15_Encrypt",
		FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;                                                                    


	/* 4. Decrypt the data using the PKCS1v15 */
	/* (Private Key)                          */
	/******************************************/       
	TST_ATP_Error = CRYS_RSA_PKCS1v15_Decrypt(TST_PrivateKey1,
		TST_PrimeData1,                                                             
		pDataOut,
		KeySize,
		pDataOut,
		&TST_ActualDecDataSize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_RSA_PKCS1v15_Decrypt",
		FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;                                                                    

	/* 6. Compare the outputted result to the input one */
	/****************************************************/
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pDataOut,
		pDataIn,
		DataSize);   

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"DX_VOS_MemCmp",
		FailureFunctionNameOut);

End:
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);
	DX_VOS_MemFree(TST_PrivateKey1);
	DX_VOS_MemFree(TST_PublicKey1);
	DX_VOS_MemFree(TST_PrimeData1);
	DX_VOS_MemFree(TST_KeyGenData1);
	

	return TST_TestStatus;	
}


/***********************************************************************
* Function Name:                                                           
*  SEPQA_RSA_PKCS1v15_NonIntegrate_SignVerify         
* 
* Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* KeyGenType - key gen type (internal/external)
* CRT_Mode - mode of the private key
* hashMode - hash mode
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
*  dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
* This function is qa external application on the SEP, 
* that executes acceptance non integrated test for RSA PKCS1v15 Sign & Verify
* Algorithm:   
* 1. Allocate memory buffers       
* 2. Prepare keys 
*     a. Generate keys 
*     or
*     a. Build the Private Key from external exponent and modulus
*     b. Build the Public Key from External exponent and modulus
* 3. Call the CRYS_RSA_PKCS1v15_SignInit to init the Sign operation
* 4. Call the CRYS_RSA_PKCS1v15_SignUpdate - This function is called N times
*                                        (N = number of blocks in data each block is 16byte)
* 5. Call the RSA_SignFinish to complete the sign operation
* 6. Call the CRYS_RSA_PKCS1v15_VerifyInit to init the verification operation
* 7. Call the CRYS_RSA_PKCS1v15_VerifyUpdate - This function is called N times
*                                        (N = number of blocks in data each block is 16byte)
* 8.Call the CRYS_RSA_PKCS1v15_VerifyFinish to complete the verification operation            * 6. Free allocated buffers
* 9. Free allocated buffers
*
**************************************************************************************/                                
CRYSError_t SEPQA_RSA_PKCS1v15_NonIntegrate_SignVerify(
								DxUint8_t	*PubExponent_ptr,
								DxUint16_t	 PubExponentSize,
								DxUint8_t   *PrivExponent_ptr,
								DxUint16_t   PrivExponentSize,
								DxUint8_t	*Modulus_ptr,
								DxUint16_t	 KeySize,
								DxUint8_t    CRT_Mode,
								DxUint8_t    KeyGenType,
								CRYS_RSA_HASH_OpMode_t hashMode,
								DxUint8_t   *P_ptr, 
								DxUint16_t	 PSize,
								DxUint8_t   *Q_ptr,
								DxUint16_t   QSize,
								DxUint8_t   *dP_ptr, 
								DxUint16_t   dPSize,
								DxUint8_t   *dQ_ptr,
								DxUint16_t   dQSize,
								DxUint8_t   *qInv_ptr,
								DxUint16_t   qInvSize,
								DxUint8_t   *Data_ptr,
								DxUint16_t   DataSize,
								DxUint8_t   *FailureFunctionNameOut)
{
	/*** TST Variables Definitions ****/
	CRYSError_t					TST_TestStatus = 0;
	DxUint8_t					Block_index = 0;
	DxUint8_t					NumOfBlocks = 0;
	DxUint8_t					*pDataIn;
	DxUint8_t					*pDataOut;
	CRYS_RSAUserPrivKey_t       *TST_PrivateKey1;
	CRYS_RSAUserPubKey_t        *TST_PublicKey1;
	CRYS_RSAKGData_t            *TST_KeyGenData1;
	CRYS_RSAPrivUserContext_t   *TST_ContextPrivate1; 
	CRYS_RSAPubUserContext_t    *TST_ContextPublic1; 

	/* 1. Allocate memory buffers */
	/******************************/
	TST_ContextPrivate1 = DX_VOS_MemMalloc(sizeof(CRYS_RSAPrivUserContext_t));
	TST_ContextPublic1 = DX_VOS_MemMalloc(sizeof(CRYS_RSAPubUserContext_t));
	TST_PrivateKey1     = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPrivKey_t));
	TST_PublicKey1      = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPubKey_t));
	TST_KeyGenData1     = DX_VOS_MemMalloc(sizeof(CRYS_RSAKGData_t));
	pDataIn				= DX_VOS_MemMalloc(DataSize);
	pDataOut			= DX_VOS_MemMalloc(KeySize);

	if ((TST_ContextPrivate1 == DX_NULL) ||
		(TST_ContextPublic1 == DX_NULL)  ||
		(TST_PrivateKey1 == DX_NULL)     ||
		(TST_PublicKey1 == DX_NULL)      ||
		(TST_KeyGenData1 == DX_NULL)     ||
		(pDataIn == DX_NULL)             ||
		(pDataOut == DX_NULL))
	{
		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
			"DX_VOS_MemMalloc",
			FailureFunctionNameOut);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, Data_ptr, DataSize);

	/*Cleaning the keys to prevent test errors with KG*/ 
	DX_VOS_MemSet(TST_PrivateKey1,0,sizeof(CRYS_RSAUserPrivKey_t));
	DX_VOS_MemSet(TST_PublicKey1,0,sizeof(CRYS_RSAUserPubKey_t));
	DX_VOS_MemSet(TST_ContextPrivate1,0,sizeof(CRYS_RSAPrivUserContext_t));
	DX_VOS_MemSet(TST_ContextPublic1,0,sizeof(CRYS_RSAPubUserContext_t));
	DX_VOS_MemSet(TST_KeyGenData1,0,sizeof(CRYS_RSAKGData_t));

	NumOfBlocks = KeySize/TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS;

	/* 2. Prepare keys */
	/*******************/
	if (KeyGenType == TST_INTERNAL_KG)
	{
		if (CRT_Mode == TST_NO_CRT)
		{
			/* Generate The Public and private keys */
			/****************************************/
			TST_ATP_Error = CRYS_RSA_KG_GenerateKeyPair(
				PubExponent_ptr,
				PubExponentSize,
				KeySize*8,
				TST_PrivateKey1,
				TST_PublicKey1,
				TST_KeyGenData1);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_KG_GenerateKeyPair",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                                    
		}
		else
		{
			/* Generate The Public and private CRT keys */
			/********************************************/
			TST_ATP_Error = CRYS_RSA_KG_GenerateKeyPairCRT(
				PubExponent_ptr,
				PubExponentSize,
				KeySize*8,
				TST_PrivateKey1,
				TST_PublicKey1,
				TST_KeyGenData1);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_KG_GenerateKeyPairCRT",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                                                                                                
		}
	}
	else
	{
		/* Build the public Key */
		/************************/
		TST_ATP_Error = CRYS_RSA_Build_PubKey(TST_PublicKey1,
			PubExponent_ptr,
			PubExponentSize,
			Modulus_ptr,
			KeySize);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
			"CRYS_RSA_Build_PubKey",
			FailureFunctionNameOut);

		if (TST_TestStatus != TESTPASS) 
			goto End;                                                                                                                                

		/* Build the Private Key */
		/*************************/ 
		if (CRT_Mode == TST_NO_CRT)
		{
			TST_ATP_Error = CRYS_RSA_Build_PrivKey(TST_PrivateKey1,
				PrivExponent_ptr,
				PrivExponentSize,
				PubExponent_ptr,
				PubExponentSize,                                                  
				Modulus_ptr,
				KeySize);                                 

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_Build_PrivKey",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                                                                                                
		}
		else
		{
			TST_ATP_Error = CRYS_RSA_Build_PrivKeyCRT(TST_PrivateKey1,
				P_ptr,
				PSize,
				Q_ptr,
				QSize,
				dP_ptr,
				dPSize,
				dQ_ptr,
				dQSize,
				qInv_ptr,
				qInvSize);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_Build_PrivKeyCRT",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                                                                                                
		}
	}/* End of TST_KeyGenType*/

	
	

	/*3. Init the Sign operation */
	/*****************************/
	TST_ATP_Error = CRYS_RSA_PKCS1v15_SignInit(TST_ContextPrivate1,
											   TST_PrivateKey1,
											   hashMode);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_RSA_PKCS1v15_SignInit",
												FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;      

	/* 4. Sign the data using the private Key */
	/* Calling the update function            */
	/******************************************/
	NumOfBlocks = DataSize/TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS;



	for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/*RSA PSS_SignUpdate block loop*/
	{ 
		TST_ATP_Error = CRYS_RSA_PKCS1v15_SignUpdate(TST_ContextPrivate1,
												pDataIn+(TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS*Block_index),
												TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t));                                        

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_RSA_PKCS1v15_SignUpdate",
													FailureFunctionNameOut);
		
		if (TST_TestStatus != TESTPASS) 
			goto End;                                                                                                                                
	}      

	

	/* 5.Finish the Sign operation */
	/*******************************/
	TST_ATP_Error = CRYS_RSA_PKCS1v15_SignFinish(TST_ContextPrivate1,
												pDataOut,
												&KeySize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_RSA_PKCS1v15_SignFinish",
		FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;                                                                                                                                


	/*6. Init the Verify operation */
	/*******************************/

	TST_ATP_Error = CRYS_RSA_PKCS1v15_VerifyInit(TST_ContextPublic1,
												 TST_PublicKey1,
												 hashMode);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_RSA_PKCS1v15_VerifyInit",
												FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;    

	
	/* 7. Verify the data using the private Key */
	/* Calling the update function              */
	/********************************************/
	for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/*RSA PSS_SignUpdate block loop*/
	{ 
		CRYS_RSA_PKCS1v15_VerifyUpdate( TST_ContextPublic1,                                     
										pDataIn+(TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS*Block_index),
										TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t));

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
			"CRYS_RSA_PKCS1v15_VerifyUpdate",
			FailureFunctionNameOut);

		if (TST_TestStatus != TESTPASS) 
			goto End;                                                                                                                                
	} 

	/* 8. Finish the Verify operation */
	/**********************************/   
	TST_ATP_Error = CRYS_RSA_PKCS1v15_VerifyFinish(TST_ContextPublic1,
													pDataOut);

	
	

	

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_RSA_PKCS1v15_VerifyFinish",
												FailureFunctionNameOut);

	

	if (TST_TestStatus != TESTPASS) 
		goto End;   

End:
	/* 9. Free allocated buffers */
	/*****************************/
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);
	DX_VOS_MemFree(TST_ContextPrivate1);
	DX_VOS_MemFree(TST_ContextPublic1);
	DX_VOS_MemFree(TST_PublicKey1);
	DX_VOS_MemFree(TST_PrivateKey1);
	DX_VOS_MemFree(TST_KeyGenData1);
	
	return TST_TestStatus;	
}


/***********************************************************************
* Function Name:                                                           
*  SEPQA_RSA_PKCS1v15_Integrate_SignVerify         
* 
* Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* KeyGenType - key gen type (internal/external)
* CRT_Mode - mode of the private key
* hashMode - hash mode
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
*  dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
* This function is qa external application on the SEP, 
* that executes acceptance non integrated test for RSA PKCS1v15 Sign & Verify
* Algorithm:   
* 1. Allocate memory buffers       
* 2. Prepare keys 
*     a. Generate keys 
*     or
*     a. Build the Private Key from external exponent and modulus
*     b. Build the Public Key from External exponent and modulus
* 3. Call the CRYS_RSA_PKCS1v15_SignInit to init the Sign operation
* 4. Call the CRYS_RSA_PKCS1v15_SignUpdate - This function is called N times
*                                        (N = number of blocks in data each block is 16byte)
* 5. Call the RSA_SignFinish to complete the sign operation
* 6. Call the CRYS_RSA_PKCS1v15_VerifyInit to init the verification operation
* 7. Call the CRYS_RSA_PKCS1v15_VerifyUpdate - This function is called N times
*                                        (N = number of blocks in data each block is 16byte)
* 8.Call the CRYS_RSA_PKCS1v15_VerifyFinish to complete the verification operation            * 6. Free allocated buffers
* 9. Free allocated buffers
*
**************************************************************************************/                                
CRYSError_t SEPQA_RSA_PKCS1v15_Integrate_SignVerify(
						DxUint8_t	*PubExponent_ptr,
						DxUint16_t	 PubExponentSize,
						DxUint8_t   *PrivExponent_ptr,
						DxUint16_t   PrivExponentSize,
						DxUint8_t	*Modulus_ptr,
						DxUint16_t	 KeySize,
						DxUint8_t    CRT_Mode,
						DxUint8_t    KeyGenType,
						CRYS_RSA_HASH_OpMode_t hashMode,
						DxUint8_t   *P_ptr, 
						DxUint16_t	 PSize,
						DxUint8_t   *Q_ptr,
						DxUint16_t   QSize,
						DxUint8_t   *dP_ptr, 
						DxUint16_t   dPSize,
						DxUint8_t   *dQ_ptr,
						DxUint16_t   dQSize,
						DxUint8_t   *qInv_ptr,
						DxUint16_t   qInvSize,
						DxUint8_t   *Data_ptr,
						DxUint16_t   DataSize,
						DxUint8_t   *FailureFunctionNameOut)
{
	/*** TST Variables Definitions ****/
	CRYSError_t					TST_TestStatus = 0;
	DxUint8_t					*pDataIn;
	DxUint8_t					*pDataOut;
	CRYS_RSAUserPrivKey_t       *TST_PrivateKey;
	CRYS_RSAUserPubKey_t        *TST_PublicKey;
	CRYS_RSAKGData_t            *TST_KeyGenData;
	CRYS_RSAPrivUserContext_t   *TST_ContextPrivate; 
	CRYS_RSAPubUserContext_t    *TST_ContextPublic; 

	/* 1. Allocate memory buffers */
	/******************************/
	TST_ContextPrivate = DX_VOS_MemMalloc(sizeof(CRYS_RSAPrivUserContext_t));
	TST_ContextPublic  = DX_VOS_MemMalloc(sizeof(CRYS_RSAPubUserContext_t));
	TST_PrivateKey     = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPrivKey_t));
	TST_PublicKey      = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPubKey_t));
	TST_KeyGenData     = DX_VOS_MemMalloc(sizeof(CRYS_RSAKGData_t));
	pDataIn			   = DX_VOS_MemMalloc(DataSize);
	pDataOut           = DX_VOS_MemMalloc(KeySize);

	if ((TST_ContextPrivate == DX_NULL) ||
		(TST_ContextPublic == DX_NULL)  ||
		(TST_PrivateKey == DX_NULL)     ||
		(TST_PublicKey == DX_NULL)      ||
		(TST_KeyGenData == DX_NULL)     ||
		(pDataIn == DX_NULL)             ||
		(pDataOut == DX_NULL))
	{
		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
													"DX_VOS_MemMalloc",
													FailureFunctionNameOut);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, Data_ptr, DataSize);

	/*Cleaning the keys to prevent test errors with KG*/ 
	DX_VOS_MemSet(TST_PrivateKey,0,sizeof(CRYS_RSAUserPrivKey_t));
	DX_VOS_MemSet(TST_PublicKey,0,sizeof(CRYS_RSAUserPubKey_t));
	DX_VOS_MemSet(TST_ContextPrivate,0,sizeof(CRYS_RSAPrivUserContext_t));
	DX_VOS_MemSet(TST_ContextPublic,0,sizeof(CRYS_RSAPubUserContext_t));
	DX_VOS_MemSet(TST_KeyGenData,0,sizeof(CRYS_RSAKGData_t));

	/* 2. Prepare keys */
	/*******************/
	if (KeyGenType == TST_INTERNAL_KG)
	{
		if (CRT_Mode == TST_NO_CRT)
		{
			/* Generate The Public and private keys */
			/****************************************/
			TST_ATP_Error = CRYS_RSA_KG_GenerateKeyPair(
				PubExponent_ptr,
				PubExponentSize,
				KeySize*8,
				TST_PrivateKey,
				TST_PublicKey,
				TST_KeyGenData);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
														"CRYS_RSA_KG_GenerateKeyPair",
														FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                                       
		}
		else
		{
			/* Generate The Public and private CRT keys */
			/****************/
			TST_ATP_Error = CRYS_RSA_KG_GenerateKeyPairCRT(
				PubExponent_ptr,
				PubExponentSize,
				KeySize*8,
				TST_PrivateKey,
				TST_PublicKey,
				TST_KeyGenData);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
														"CRYS_RSA_KG_GenerateKeyPairCRT",
														FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                                                                                                   
		}
	}
	else
	{                    
		/* Build the public Key */
		/************************/
		TST_ATP_Error = CRYS_RSA_Build_PubKey(TST_PublicKey,
												PubExponent_ptr,
												PubExponentSize,
												Modulus_ptr,
												KeySize);


		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_RSA_Build_PubKey",
													FailureFunctionNameOut);

		if (TST_TestStatus != TESTPASS) 
			goto End;                                                                                                                                   

		/* Build the Private Key */
		/*************************/ 
		if (CRT_Mode == TST_NO_CRT)
		{
			TST_ATP_Error = CRYS_RSA_Build_PrivKey( TST_PrivateKey,
													PrivExponent_ptr,
													PrivExponentSize,
													PubExponent_ptr,
													PubExponentSize,
													Modulus_ptr,
													KeySize);                                 

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
														"CRYS_RSA_Build_PrivKey",
														FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                                                                                                   
		}
		else
		{
			TST_ATP_Error = CRYS_RSA_Build_PrivKeyCRT(TST_PrivateKey,
														P_ptr,
														PSize,
														Q_ptr,
														QSize,
														dP_ptr,
														dPSize,
														dQ_ptr,
														dQSize,
														qInv_ptr,
														qInvSize);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_Build_PrivKeyCRT",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;
		}
	}/*End of TST_KeyGenType*/

	

	/*3. Preform the Sign operation */
	/********************************/
	TST_ATP_Error = CRYS_RSA_PKCS1v15_Sign(TST_ContextPrivate,
										   TST_PrivateKey,
										   hashMode,
										   pDataIn,
										   DataSize,
										   pDataOut,
										   &KeySize);                                            

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_RSA_PKCS1v15_Sign",
		FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/*4. Preform the Verify operation */
	/**********************************/
	TST_ATP_Error = CRYS_RSA_PKCS1v15_Verify(TST_ContextPublic,
											 TST_PublicKey,
											 hashMode,
											 pDataIn,
											 DataSize,
											 pDataOut);                                                

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
											"CRYS_RSA_PKCS1v15_Verify",
											FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;

End:
	/* 9. Free allocated buffers */
	/*****************************/
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);
	DX_VOS_MemFree(TST_ContextPrivate);
	DX_VOS_MemFree(TST_ContextPublic);
	DX_VOS_MemFree(TST_PublicKey);
	DX_VOS_MemFree(TST_PrivateKey);
	DX_VOS_MemFree(TST_KeyGenData);

	return TST_TestStatus;	
}

/***********************************************************************
* Function Name:                                                           
*  SEPQA_RSA_OAEP_EncDec         
* 
* Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* KeyGenType - key gen type (internal/external)
* CRT_Mode - mode of the private key
* hashMode - hash mode
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
*  dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
* This function is qa external application on the SEP, 
* that executes acceptance test for RSA OAEP encrypt and decrypt
* according to PKCS#1 Ver2.1
* Algorithm:   
* 1. Allocate memory buffers       
* 2. Prepare keys 
*     a. Generate keys 
*     or
*     a. Build the Private Key from external exponent and modulus
*     b. Build the Public Key from External exponent and modulus       
* 3. Encrypt a known vector
* 4. Decrypt the output of the previous step
* 5. Compare the outputted result to the input one                  
* 6. Free allocated buffers
*
**************************************************************************************/                                

CRYSError_t SEPQA_RSA_OAEP_EncDec(
			DxUint8_t	*PubExponent_ptr,
			DxUint16_t	 PubExponentSize,
			DxUint8_t   *PrivExponent_ptr,
			DxUint16_t   PrivExponentSize,
			DxUint8_t	*Modulus_ptr,
			DxUint16_t	 KeySize,
			DxUint8_t    CRT_Mode,
			DxUint8_t    KeyGenType,
			CRYS_RSA_HASH_OpMode_t hashMode,
			DxUint8_t   *P_ptr, 
			DxUint16_t	 PSize,
			DxUint8_t   *Q_ptr,
			DxUint16_t   QSize,
			DxUint8_t   *dP_ptr, 
			DxUint16_t   dPSize,
			DxUint8_t   *dQ_ptr,
			DxUint16_t   dQSize,
			DxUint8_t   *qInv_ptr,
			DxUint16_t   qInvSize,
			DxUint8_t   *Data_ptr,
			DxUint16_t   DataSize,
			DxUint8_t   *FailureFunctionNameOut)
{
	/*** TST Variables Definitions ****/
	CRYSError_t					TST_TestStatus = 0;
	DxUint16_t					TST_ActualDecDataSize = 0;
	DxUint8_t					*pDataIn;
	DxUint8_t					*pDataOut;
	CRYS_RSAUserPrivKey_t       *TST_PrivateKey1;
	CRYS_RSAUserPubKey_t        *TST_PublicKey1;
	CRYS_RSAPrimeData_t         *TST_PrimeData1;
	CRYS_RSAKGData_t            *TST_KeyGenData1;

	/* 1. Allocate memory buffers */
	/******************************/
	pDataIn			= DX_VOS_MemMalloc(DataSize);
	pDataOut		= DX_VOS_MemMalloc(KeySize);
	TST_PrivateKey1 = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPrivKey_t));
	TST_PublicKey1	= DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPubKey_t));
	TST_KeyGenData1 = DX_VOS_MemMalloc(sizeof(CRYS_RSAKGData_t));
	TST_PrimeData1	= DX_VOS_MemMalloc(sizeof(CRYS_RSAPrimeData_t));	

	if ((TST_PrivateKey1 == DX_NULL)     ||
		(TST_PublicKey1  == DX_NULL)     ||
		(TST_KeyGenData1 == DX_NULL)     ||
		(TST_PrimeData1  == DX_NULL)     ||
		(pDataIn         == DX_NULL)     ||
		(pDataOut        == DX_NULL))
	{
		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
			"DX_VOS_MemMalloc",
			FailureFunctionNameOut);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, Data_ptr, DataSize);

	TST_ActualDecDataSize = KeySize*8;

	/*Cleaning the keys to prevent test errors with KG*/ 
	DX_VOS_MemSet(TST_PrivateKey1,0,sizeof(CRYS_RSAUserPrivKey_t));
	DX_VOS_MemSet(TST_PublicKey1,0,sizeof(CRYS_RSAUserPubKey_t));
	DX_VOS_MemSet(TST_KeyGenData1,0,sizeof(CRYS_RSAKGData_t));
	DX_VOS_MemSet(TST_PrimeData1,0,sizeof(CRYS_RSAPrimeData_t));

	/* 2. Prepare keys */
	/*******************/
	if (KeyGenType == TST_INTERNAL_KG)
	{
		if (CRT_Mode == TST_NO_CRT)
		{
			/* Generate The Public and private keys */
			/****************************************/
			TST_ATP_Error = CRYS_RSA_KG_GenerateKeyPair(
				PubExponent_ptr,
				PubExponentSize,
				KeySize*8,
				TST_PrivateKey1,
				TST_PublicKey1,
				TST_KeyGenData1);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_KG_GenerateKeyPair",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                                       
		}
		else
		{
			/* Generate The Public and private CRT keys */
			/****************/
			TST_ATP_Error = CRYS_RSA_KG_GenerateKeyPairCRT(
				PubExponent_ptr,
				PubExponentSize,
				KeySize*8,
				TST_PrivateKey1,
				TST_PublicKey1,
				TST_KeyGenData1);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_KG_GenerateKeyPairCRT",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                                                                                                   
		}
	}
	else
	{                    
		/* Build the public Key */
		/************************/
		TST_ATP_Error = CRYS_RSA_Build_PubKey(TST_PublicKey1,
											  PubExponent_ptr,
											  PubExponentSize,
											  Modulus_ptr,
											  KeySize);


		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
			"CRYS_RSA_Build_PubKey",
			FailureFunctionNameOut);

		if (TST_TestStatus != TESTPASS) 
			goto End;                                                                                                                                   

		/* Build the Private Key */
		/*************************/ 
		if (CRT_Mode == TST_NO_CRT)
		{
			TST_ATP_Error = CRYS_RSA_Build_PrivKey( TST_PrivateKey1,
													PrivExponent_ptr,
													PrivExponentSize,
													PubExponent_ptr,
													PubExponentSize,
													Modulus_ptr,
													KeySize);                                 

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_Build_PrivKey",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                                                                                                   
		}
		else
		{
			TST_ATP_Error = CRYS_RSA_Build_PrivKeyCRT(TST_PrivateKey1,
													P_ptr,
													PSize,
													Q_ptr,
													QSize,
													dP_ptr,
													dPSize,
													dQ_ptr,
													dQSize,
													qInv_ptr,
													qInvSize);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_Build_PrivKeyCRT",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;
		}
	}/*End of TST_KeyGenType*/

	/* 3. Encrypt the data using the OAEP */
	/* (Public Key)                       */
	/**************************************/   
	TST_ATP_Error = CRYS_RSA_OAEP_Encrypt(TST_PublicKey1,
										  TST_PrimeData1,
										  hashMode,
										  DX_NULL,
										  TST_ZERO_DATA,
										  CRYS_PKCS1_MGF1,
										  pDataIn,
										  DataSize,
										  pDataOut);                                                                                                      

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_RSA_OAEP_Encrypt",
												FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 4. Decrypt the data using the OAEP */
	/* (Private Key)                      */
	/**************************************/       
	TST_ATP_Error = CRYS_RSA_OAEP_Decrypt(TST_PrivateKey1,
										  TST_PrimeData1,
										  hashMode,               
										  DX_NULL,
										  TST_ZERO_DATA,
										  CRYS_PKCS1_MGF1,                                            
										  pDataOut,
										  KeySize,
										  pDataOut,
										  &TST_ActualDecDataSize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_RSA_OAEP_Decrypt",
												FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 5. Compare the outputted result to the input one */
	/****************************************************/
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pDataOut,
											   pDataIn,
											   DataSize);   

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"DX_VOS_MemCmp",
												FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;

End:
	/* 9. Free allocated buffers */
	/*****************************/
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);
	DX_VOS_MemFree(TST_PrivateKey1);
	DX_VOS_MemFree(TST_PublicKey1);
	DX_VOS_MemFree(TST_PrimeData1);
	DX_VOS_MemFree(TST_KeyGenData1);

	return TST_TestStatus;	
}

/***********************************************************************
* Function Name:                                                           
*  SEPQA_RSA_PSS_NonIntegrate_SignVerify         
* 
* Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* KeyGenType - key gen type (internal/external)
* CRT_Mode - mode of the private key
* hashMode - hash mode
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
*  dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
* This function is qa external application on the SEP, 
* that executes acceptance test for RSA PSS PKCS#1v2.1 sign & Verify
* Algorithm:   
* 1. Allocate memory buffers       
* 2. Prepare keys 
*     a. Generate keys 
*     or
*     a. Build the Private Key from external exponent and modulus
*     b. Build the Public Key from External exponent and modulus   
* 3. Call the CRYS_RSA_PSS_SignInit to init the Sign operation
* 4. Call the CRYS_RSA_PSS_SignUpdate - This function is called N times
*                                        (N = number of blocks in data each block is 16byte)
* 5. Call the RSA_SignFinish to complete the sign operation
* 6. Call the CRYS_RSA_PSS_VerifyInit to init the verification operation
* 7. Call the CRYS_RSA_PSS_VerifyUpdate - This function is called N times
*                                        (N = number of blocks in data each block is 16byte)
* 8.Call the CRYS_RSA_PSS_VerifyFinish to complete the verification operation             
* 9. Free allocated buffers
*
**************************************************************************************/                                

CRYSError_t SEPQA_RSA_PSS_NonIntegrate_SignVerify(
								  DxUint8_t	            *PubExponent_ptr,
								  DxUint16_t	         PubExponentSize,
								  DxUint8_t             *PrivExponent_ptr,
								  DxUint16_t             PrivExponentSize,
								  DxUint8_t	            *Modulus_ptr,
								  DxUint16_t	         KeySize,
								  DxUint8_t              CRT_Mode,
								  DxUint8_t				 KeyGenType,
								  CRYS_RSA_HASH_OpMode_t HashMode,
								  DxUint16_t		     SaltLength,
								  DxUint8_t             *P_ptr, 
								  DxUint16_t	         PSize,
								  DxUint8_t             *Q_ptr,
								  DxUint16_t             QSize,
								  DxUint8_t             *dP_ptr, 
								  DxUint16_t             dPSize,
								  DxUint8_t             *dQ_ptr,
								  DxUint16_t             dQSize,
								  DxUint8_t             *qInv_ptr,
								  DxUint16_t             qInvSize,
								  DxUint8_t             *Data_ptr,
								  DxUint16_t             DataSize,
								  DxUint8_t             *FailureFunctionNameOut)
{
	/*** TST Variables Definitions ****/
	CRYSError_t					TST_TestStatus = 0;
	DxUint16_t					Block_index = 0;
	DxUint16_t					NumOfBlocks = 0;
	DxUint8_t					*pDataIn = DX_NULL;
	DxUint8_t					*pDataOut = DX_NULL;
	CRYS_RSAUserPrivKey_t       *TST_PrivateKey1 = DX_NULL;
	CRYS_RSAUserPubKey_t        *TST_PublicKey1 = DX_NULL;
	CRYS_RSAPrimeData_t         *TST_PrimeData1 = DX_NULL;
	CRYS_RSAKGData_t            *TST_KeyGenData1 = DX_NULL;
	CRYS_RSAPrivUserContext_t   *TST_ContextPrivate1 = DX_NULL; 
	CRYS_RSAPubUserContext_t    *TST_ContextPublic1 = DX_NULL; 
	
	/* 1. Allocate memory buffers */
	/******************************/
	TST_ContextPrivate1 = DX_VOS_MemMalloc(sizeof(CRYS_RSAPrivUserContext_t));
	TST_ContextPublic1  = DX_VOS_MemMalloc(sizeof(CRYS_RSAPubUserContext_t));
	TST_PrivateKey1     = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPrivKey_t));
	TST_PublicKey1      = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPubKey_t));
	TST_KeyGenData1     = DX_VOS_MemMalloc(sizeof(CRYS_RSAKGData_t));
	TST_PrimeData1      = DX_VOS_MemMalloc(sizeof(CRYS_RSAPrimeData_t));
	pDataIn             = DX_VOS_MemMalloc(DataSize);
	pDataOut            = DX_VOS_MemMalloc(KeySize);

	if ((TST_ContextPrivate1 == DX_NULL) ||
		(TST_ContextPublic1 == DX_NULL)  ||
		(TST_PrivateKey1 == DX_NULL)     ||
		(TST_PublicKey1 == DX_NULL)      ||
		(TST_KeyGenData1 == DX_NULL)     ||
		(TST_PrimeData1 == DX_NULL)      ||
		(pDataIn == DX_NULL)             ||
		(pDataOut == DX_NULL))
	{
		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
													"DX_VOS_MemMalloc",
													FailureFunctionNameOut);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, Data_ptr, DataSize);

	/*Cleaning the keys to prevent test errors with KG*/ 
	DX_VOS_MemSet(TST_PrivateKey1,0,sizeof(CRYS_RSAUserPrivKey_t));
	DX_VOS_MemSet(TST_PublicKey1,0,sizeof(CRYS_RSAUserPubKey_t));
	DX_VOS_MemSet(TST_KeyGenData1,0,sizeof(CRYS_RSAKGData_t));
	DX_VOS_MemSet(TST_PrimeData1,0,sizeof(CRYS_RSAPrimeData_t));
	DX_VOS_MemSet(TST_ContextPrivate1,0,sizeof(CRYS_RSAPrivUserContext_t));
	DX_VOS_MemSet(TST_ContextPublic1,0,sizeof(CRYS_RSAPubUserContext_t));

	NumOfBlocks = KeySize/TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS;          

	/* 2. Prepare keys */
	/*******************/
	if (KeyGenType == TST_INTERNAL_KG)
	{
		if (CRT_Mode == TST_NO_CRT)
		{
			/* Generate The Public and private keys */
			/****************************************/
			TST_ATP_Error = CRYS_RSA_KG_GenerateKeyPair(
				PubExponent_ptr,
				PubExponentSize,
				KeySize*8,
				TST_PrivateKey1,
				TST_PublicKey1,
				TST_KeyGenData1);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_KG_GenerateKeyPair",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                                       
		}
		else
		{
			/* Generate The Public and private CRT keys */
			/****************/
			TST_ATP_Error = CRYS_RSA_KG_GenerateKeyPairCRT(
				PubExponent_ptr,
				PubExponentSize,
				KeySize*8,
				TST_PrivateKey1,
				TST_PublicKey1,
				TST_KeyGenData1);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_KG_GenerateKeyPairCRT",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                                                                                                   
		}
	}
	else
	{                    
		/* Build the public Key */
		/************************/
		TST_ATP_Error = CRYS_RSA_Build_PubKey(TST_PublicKey1,
			PubExponent_ptr,
			PubExponentSize,
			Modulus_ptr,
			KeySize);


		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
			"CRYS_RSA_Build_PubKey",
			FailureFunctionNameOut);

		if (TST_TestStatus != TESTPASS) 
			goto End;                                                                                                                                   

		/* Build the Private Key */
		/*************************/ 
		if (CRT_Mode == TST_NO_CRT)
		{
			TST_ATP_Error = CRYS_RSA_Build_PrivKey(TST_PrivateKey1,
				PrivExponent_ptr,
				PrivExponentSize,
				PubExponent_ptr,
				PubExponentSize,
				Modulus_ptr,
				KeySize);                                 

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_Build_PrivKey",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                                                                                                   
		}
		else
		{
			TST_ATP_Error = CRYS_RSA_Build_PrivKeyCRT(TST_PrivateKey1,
				P_ptr,
				PSize,
				Q_ptr,
				QSize,
				dP_ptr,
				dPSize,
				dQ_ptr,
				dQSize,
				qInv_ptr,
				qInvSize);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_Build_PrivKeyCRT",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;
		}
	}/*End of TST_KeyGenType*/

	/*3. Init the Sign operation */
	/*****************************/
	TST_ATP_Error = CRYS_RSA_PSS_SignInit(TST_ContextPrivate1,
										  TST_PrivateKey1,
										  HashMode,
										  CRYS_PKCS1_MGF1,
										  SaltLength);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_RSA_PSS_SignInit",
												FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 4. Sign the data using the private Key */
	/* Calling the update function            */
	/******************************************/
	NumOfBlocks = DataSize/TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS;

	for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/*RSA PSS_SignUpdate block loop*/
	{ 
		TST_ATP_Error = CRYS_RSA_PSS_SignUpdate(TST_ContextPrivate1,
												pDataIn+(TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS*Block_index),
												TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t));                                        

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_RSA_PSS_SignUpdate",
													FailureFunctionNameOut);

		if (TST_TestStatus != TESTPASS) 
			goto End;
	}                       


	/* 5. Finish the Sign operation */
	/********************************/ 
	TST_ATP_Error = CRYS_RSA_PSS_SignFinish(TST_ContextPrivate1,
											pDataOut,
											&KeySize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_RSA_PSS_SignFinish",
												FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;


	/*6. Init the Verify operation */
	/*--------------------------*/
	TST_ATP_Error = CRYS_RSA_PSS_VerifyInit(TST_ContextPublic1,
											TST_PublicKey1,
											HashMode,
											CRYS_PKCS1_MGF1,
											SaltLength);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_RSA_PSS_VerifyInit",
												FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 7. Verify the data using the private Key */
	/* Calling the update function              */
	/********************************************/
	for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/*RSA PSS_SignUpdate block loop*/
	{ 
		TST_ATP_Error = CRYS_RSA_PSS_VerifyUpdate(TST_ContextPublic1,
										pDataIn+(TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS*Block_index),
										TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t));                                        

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_RSA_PSS_VerifyUpdate",
													FailureFunctionNameOut);

		if (TST_TestStatus != TESTPASS) 
			goto End;
	}     

	/* 8. Finish the Verify operation */
	/**********************************/   
	TST_ATP_Error = CRYS_RSA_PSS_VerifyFinish(TST_ContextPublic1,
											  pDataOut);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_RSA_PSS_VerifyFinish",
												FailureFunctionNameOut);

End:
	/* 9. Free allocated buffers */
	/*****************************/
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);
	DX_VOS_MemFree(TST_PrivateKey1);
	DX_VOS_MemFree(TST_PublicKey1);
	DX_VOS_MemFree(TST_PrimeData1);
	DX_VOS_MemFree(TST_KeyGenData1);
	DX_VOS_MemFree(TST_ContextPrivate1);
	DX_VOS_MemFree(TST_ContextPublic1);

	return TST_TestStatus;	
}

/***********************************************************************
* Function Name:                                                           
*  SEPQA_RSA_PSS_Integrate_SignVerify         
* 
* Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* KeyGenType - key gen type (internal/external)
* CRT_Mode - mode of the private key
* hashMode - hash mode
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
*  dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
* This function is qa external application on the SEP, 
* that executes acceptance test for RSA PSS PKCS#1v2.1 sign & Verify
* Algorithm:   
* 1. Allocate memory buffers       
* 2. Prepare keys 
*     a. Generate keys 
*     or
*     a. Build the Private Key from external exponent and modulus
*     b. Build the Public Key from External exponent and modulus   
* 3. Preform the Sign operation
* 4. Perform the Verify operation
* 9. Free allocated buffers
*
**************************************************************************************/                                
CRYSError_t SEPQA_RSA_PSS_Integrate_SignVerify(
					DxUint8_t	            *PubExponent_ptr,
					DxUint16_t				PubExponentSize,
					DxUint8_t				*PrivExponent_ptr,
					DxUint16_t				PrivExponentSize,
					DxUint8_t	            *Modulus_ptr,
					DxUint16_t				KeySize,
					DxUint8_t				CRT_Mode,
					DxUint8_t				KeyGenType,
					CRYS_RSA_HASH_OpMode_t	HashMode,
					DxUint16_t				SaltLength,
					DxUint8_t				*P_ptr, 
					DxUint16_t				PSize,
					DxUint8_t				*Q_ptr,
					DxUint16_t				QSize,
					DxUint8_t				*dP_ptr, 
					DxUint16_t				dPSize,
					DxUint8_t				*dQ_ptr,
					DxUint16_t				dQSize,
					DxUint8_t				*qInv_ptr,
					DxUint16_t				qInvSize,
					DxUint8_t				*Data_ptr,
					DxUint16_t				DataSize,
					DxUint8_t				*FailureFunctionNameOut)
{
	/*** TST Variables Definitions ****/
	CRYSError_t					TST_TestStatus = 0;
	DxUint8_t					*pDataIn;
	DxUint8_t					*pDataOut;
	CRYS_RSAUserPrivKey_t       *TST_PrivateKey1;
	CRYS_RSAUserPubKey_t        *TST_PublicKey1;
	CRYS_RSAPrimeData_t         *TST_PrimeData1;
	CRYS_RSAKGData_t            *TST_KeyGenData1;
	CRYS_RSAPrivUserContext_t   *TST_ContextPrivate1; 
	CRYS_RSAPubUserContext_t    *TST_ContextPublic1; 

	/* 1. Allocate memory buffers */
	/******************************/
	TST_ContextPrivate1 = DX_VOS_MemMalloc(sizeof(CRYS_RSAPrivUserContext_t));
	TST_ContextPublic1 = DX_VOS_MemMalloc(sizeof(CRYS_RSAPubUserContext_t));
	TST_PrivateKey1     = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPrivKey_t));
	TST_PublicKey1      = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPubKey_t));
	TST_KeyGenData1     = DX_VOS_MemMalloc(sizeof(CRYS_RSAKGData_t));
	TST_PrimeData1      = DX_VOS_MemMalloc(sizeof(CRYS_RSAPrimeData_t));

	pDataIn = DX_VOS_MemMalloc(DataSize);
	pDataOut = DX_VOS_MemMalloc(KeySize);

	if ((TST_ContextPrivate1 == DX_NULL) ||
		(TST_PrimeData1 == DX_NULL)      ||
		(TST_ContextPublic1 == DX_NULL)  ||
		(TST_PrivateKey1 == DX_NULL)     ||
		(TST_PublicKey1 == DX_NULL)      ||
		(TST_KeyGenData1 == DX_NULL)     ||
		(pDataIn == DX_NULL)             ||
		(pDataOut == DX_NULL))
	{
		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
													"DX_VOS_MemMalloc",
													FailureFunctionNameOut);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, Data_ptr, DataSize);

	/*Cleaning the keys to prevent test errors with KG*/ 
	DX_VOS_MemSet(TST_PrivateKey1,0,sizeof(CRYS_RSAUserPrivKey_t));
	DX_VOS_MemSet(TST_PublicKey1,0,sizeof(CRYS_RSAUserPubKey_t));
	DX_VOS_MemSet(TST_KeyGenData1,0,sizeof(CRYS_RSAKGData_t));
	DX_VOS_MemSet(TST_PrimeData1,0,sizeof(CRYS_RSAPrimeData_t));
	DX_VOS_MemSet(TST_ContextPrivate1,0,sizeof(CRYS_RSAPrivUserContext_t));
	DX_VOS_MemSet(TST_ContextPublic1,0,sizeof(CRYS_RSAPubUserContext_t));

	/* 2. Prepare keys */
	/*******************/
	if (KeyGenType == TST_INTERNAL_KG)
	{
		if (CRT_Mode == TST_NO_CRT)
		{
			/* Generate The Public and private keys */
			/****************************************/
			TST_ATP_Error = CRYS_RSA_KG_GenerateKeyPair(
				PubExponent_ptr,
				PubExponentSize,
				KeySize*8,
				TST_PrivateKey1,
				TST_PublicKey1,
				TST_KeyGenData1);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_KG_GenerateKeyPair",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                                       
		}
		else
		{
			/* Generate The Public and private CRT keys */
			/****************/
			TST_ATP_Error = CRYS_RSA_KG_GenerateKeyPairCRT(
				PubExponent_ptr,
				PubExponentSize,
				KeySize*8,
				TST_PrivateKey1,
				TST_PublicKey1,
				TST_KeyGenData1);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_KG_GenerateKeyPairCRT",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                                                                                                   
		}
	}
	else
	{                    
		/* Build the public Key */
		/************************/
		TST_ATP_Error = CRYS_RSA_Build_PubKey(TST_PublicKey1,
			PubExponent_ptr,
			PubExponentSize,
			Modulus_ptr,
			KeySize);


		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
			"CRYS_RSA_Build_PubKey",
			FailureFunctionNameOut);

		if (TST_TestStatus != TESTPASS) 
			goto End;                                                                                                                                   

		/* Build the Private Key */
		/*************************/ 
		if (CRT_Mode == TST_NO_CRT)
		{
			TST_ATP_Error = CRYS_RSA_Build_PrivKey(TST_PrivateKey1,
				PrivExponent_ptr,
				PrivExponentSize,
				PubExponent_ptr,
				PubExponentSize,
				Modulus_ptr,
				KeySize);                                 

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_Build_PrivKey",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                                                                                                   
		}
		else
		{
			TST_ATP_Error = CRYS_RSA_Build_PrivKeyCRT(TST_PrivateKey1,
				P_ptr,
				PSize,
				Q_ptr,
				QSize,
				dP_ptr,
				dPSize,
				dQ_ptr,
				dQSize,
				qInv_ptr,
				qInvSize);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RSA_Build_PrivKeyCRT",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;
		}
	}/*End of TST_KeyGenType*/
	
	/*3. Preform the Sign operation */
	/********************************/
	TST_ATP_Error = CRYS_RSA_PSS_Sign(TST_ContextPrivate1,
									  TST_PrivateKey1,
									  HashMode,
									  CRYS_PKCS1_MGF1,
									  SaltLength,
									  pDataIn,
									  DataSize,
									  pDataOut,
									  &KeySize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_RSA_PSS_Sign",
		FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/*4. Preform the Verify operation */
	/**********************************/
	TST_ATP_Error = CRYS_RSA_PSS_Verify(TST_ContextPublic1,
										TST_PublicKey1,
										HashMode,
										CRYS_PKCS1_MGF1,
										SaltLength,
										pDataIn,
										DataSize,
										pDataOut);                                        

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_RSA_PSS_Verify",
												FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;

End:
	/* 5. Free allocated buffers */
	/*****************************/
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);
	DX_VOS_MemFree(TST_PrivateKey1);
	DX_VOS_MemFree(TST_PublicKey1);
	DX_VOS_MemFree(TST_PrimeData1);
	DX_VOS_MemFree(TST_KeyGenData1);
	DX_VOS_MemFree(TST_ContextPrivate1);
	DX_VOS_MemFree(TST_ContextPublic1);
	
	return TST_TestStatus;	
}
/***************************
* Function Name:                                                           
*  SEPQA_RSA_PKCS1v15_ExternalHash_SignVerify        
* 
* Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* KeyGenType - key gen type (internal/external)
* CRT_Mode - mode of the private key
* hashMode - hash mode
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
*  dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
* FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for RSA PKCS1v15 Sign & Verify
*  When the Hash for the Sign is preformed externally                   
*  This function Signs & Verifies hash vectors using RSA PKCS1v15
*  Vectors sizes used (in byte) - SHA1 - 20, MD5 - 16                                                      
*  Key sizes used     (in bits) - 2048
* Algorithm:       
*  1. Prepare keys 
*     a. Generate keys 
*     or
*     a. Build the Private Key from external exponent and modulus
*     b. Build the Public Key from External exponent and modulus
*  2. Call the CRYS_RSA_PKCS1v15_SHA1_Sign or CRYS_RSA_PKCS1v15_MD5_Sign to preform the Sign operation
*  3. Call the CRYS_RSA_PKCS1v15_SHA1_Verify or CRYS_RSA_PKCS1v15_MD5_verify to complete the verification operation
*  4. Compare the outputted result to the expected one                  
*
**************************/
CRYSError_t SEPQA_RSA_PKCS1v15_ExternalHash_SignVerify(DxUint8_t	 *PubExponent_ptr,
													   DxUint16_t	 PubExponentSize,
													   DxUint8_t     *PrivExponent_ptr,
													   DxUint8_t	 *Modulus_ptr,
													   DxUint16_t	 KeySize,
													   DxUint8_t     CRT_Mode,
													   DxUint8_t     KeyGenType,
													   CRYS_RSA_HASH_OpMode_t hashMode,
													   DxUint8_t     *P_ptr, 
													   DxUint16_t	 PSize,
													   DxUint8_t     *Q_ptr,
													   DxUint16_t    QSize,
													   DxUint8_t     *dP_ptr, 
													   DxUint16_t    dPSize,
													   DxUint8_t     *dQ_ptr,
													   DxUint16_t    dQSize,
													   DxUint8_t     *qInv_ptr,
													   DxUint16_t    qInvSize,
													   DxUint8_t     *Data_ptr,
													   DxUint16_t    DataSize,
													   DxUint8_t     *FailureFunctionNameOut)

{
	/*** TST Variables Definitions ****/
	CRYSError_t					TST_TestStatus = 0;
	DxUint8_t					*pDataIn;
	DxUint8_t					*pSignOut;
	CRYS_RSAUserPrivKey_t       *TST_PrivateKey1;
	CRYS_RSAUserPubKey_t        *TST_PublicKey1;
	CRYS_RSAPrimeData_t         *TST_PrimeData1;
	CRYS_RSAKGData_t            *TST_KeyGenData1;
	CRYS_RSAPrivUserContext_t   *TST_ContextPrivate1; 
	CRYS_RSAPubUserContext_t    *TST_ContextPublic1; 

	/* 1. Allocate memory buffers */
	/******************************/
	TST_ContextPrivate1 = DX_VOS_MemMalloc(sizeof(CRYS_RSAPrivUserContext_t));
	TST_ContextPublic1 = DX_VOS_MemMalloc(sizeof(CRYS_RSAPubUserContext_t));
	TST_PrivateKey1     = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPrivKey_t));
	TST_PublicKey1      = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPubKey_t));
	TST_KeyGenData1     = DX_VOS_MemMalloc(sizeof(CRYS_RSAKGData_t));
	TST_PrimeData1      = DX_VOS_MemMalloc(sizeof(CRYS_RSAPrimeData_t));

	pDataIn = DX_VOS_MemMalloc(DataSize);
	pSignOut = DX_VOS_MemMalloc(TST_MAX_PRVT_EXP_SIZE_IN_BYTES);

	if ((TST_ContextPrivate1 == DX_NULL) ||
		(TST_PrimeData1 == DX_NULL)      ||
		(TST_ContextPublic1 == DX_NULL)  ||
		(TST_PrivateKey1 == DX_NULL)     ||
		(TST_PublicKey1 == DX_NULL)      ||
		(TST_KeyGenData1 == DX_NULL)     ||
		(pDataIn == DX_NULL)             ||
		(pSignOut == DX_NULL))
	{
		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
													"DX_VOS_MemMalloc",
													FailureFunctionNameOut);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, Data_ptr, DataSize);

	/*Cleaning the keys to prevent test errors with KG*/ 
	DX_VOS_MemSet(TST_PrivateKey1,0,sizeof(CRYS_RSAUserPrivKey_t));
	DX_VOS_MemSet(TST_PublicKey1,0,sizeof(CRYS_RSAUserPubKey_t));
	DX_VOS_MemSet(TST_KeyGenData1,0,sizeof(CRYS_RSAKGData_t));
	DX_VOS_MemSet(TST_PrimeData1,0,sizeof(CRYS_RSAPrimeData_t));
	DX_VOS_MemSet(TST_ContextPrivate1,0,sizeof(CRYS_RSAPrivUserContext_t));
	DX_VOS_MemSet(TST_ContextPublic1,0,sizeof(CRYS_RSAPubUserContext_t));

	/* 2. Prepare keys */
	/*******************/
	if (KeyGenType == TST_INTERNAL_KG)
	{
		if (CRT_Mode == TST_NO_CRT)
		{
			/* Generate The Public and private keys */
			/**************/
			TST_ATP_Error = CRYS_RSA_KG_GenerateKeyPair(PubExponent_ptr,
														PubExponentSize,
														KeySize*8,
														TST_PrivateKey1,
														TST_PublicKey1,
														TST_KeyGenData1);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
														"CRYS_RSA_KG_GenerateKeyPair",
														FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                   
		}
		else
		{
			/* Generate The Public and private CRT keys */
			/****************/
			TST_ATP_Error = CRYS_RSA_KG_GenerateKeyPairCRT(PubExponent_ptr,
														   PubExponentSize,
														   KeySize*8,
														   TST_PrivateKey1,
														   TST_PublicKey1,
														   TST_KeyGenData1);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
														"CRYS_RSA_KG_GenerateKeyPairCRT",
														FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;                                                              
		}
	}
	else
	{
		/* Build the public Key */
		/*----------------------*/
		TST_ATP_Error = CRYS_RSA_Build_PubKey(TST_PublicKey1,
											  PubExponent_ptr,
											  PubExponentSize,
											  Modulus_ptr,
											  KeySize);


		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_RSA_Build_PubKey",
													FailureFunctionNameOut);

		if (TST_TestStatus != TESTPASS) 
			goto End;    

		/* Build the Private Key */
		/*-----------------------*/ 
		if (CRT_Mode == TST_NO_CRT)
		{
			TST_ATP_Error = CRYS_RSA_Build_PrivKey( TST_PrivateKey1,
													PrivExponent_ptr,
													KeySize,
													PubExponent_ptr,
													PubExponentSize,
													Modulus_ptr,
													KeySize);                                  

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
														"CRYS_RSA_Build_PrivKey",
														FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End;   
		}
		else
		{
			TST_ATP_Error = CRYS_RSA_Build_PrivKeyCRT(TST_PrivateKey1,
													  P_ptr, PSize,
													  Q_ptr, QSize,
													  dP_ptr, dPSize, 
													  dQ_ptr, dQSize,                                                                                                
													  qInv_ptr, qInvSize);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
														"CRYS_RSA_Build_PrivKeyCRT",
														FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End; 
		}
	}/* End of TST_KeyGenType*/


	/*3. Preform the Sign operation */
	/********************************/
	switch(hashMode)
	{
	case CRYS_HASH_SHA1_mode:
		TST_ATP_Error = CRYS_RSA_PKCS1v15_SHA1_Sign(TST_ContextPrivate1,
													TST_PrivateKey1,
													pDataIn,
													pSignOut,
													&KeySize); 
		break;
	case CRYS_HASH_SHA224_mode:
		TST_ATP_Error = CRYS_RSA_PKCS1v15_SHA224_Sign(TST_ContextPrivate1,
													  TST_PrivateKey1,
													  pDataIn,
													  pSignOut,
													  &KeySize); 
		break;
	case CRYS_HASH_SHA256_mode:
		TST_ATP_Error = CRYS_RSA_PKCS1v15_SHA256_Sign(TST_ContextPrivate1,
													  TST_PrivateKey1,
													  pDataIn,
													  pSignOut,
													  &KeySize); 
		break;
	case CRYS_HASH_SHA384_mode:
#ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
		TST_ATP_Error = CRYS_RSA_PKCS1v15_SHA384_Sign(TST_ContextPrivate1,
													  TST_PrivateKey1,
													  pDataIn,
													  pSignOut,
													  &KeySize); 
		break;
#else
		goto End;
#endif
	case CRYS_HASH_SHA512_mode:
#ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
		TST_ATP_Error = CRYS_RSA_PKCS1v15_SHA512_Sign(TST_ContextPrivate1,
													  TST_PrivateKey1,
													  pDataIn,
													  pSignOut,
													  &KeySize);
		break;
#else
		goto End;
#endif
	case CRYS_HASH_MD5_mode:    
		TST_ATP_Error = CRYS_RSA_PKCS1v15_MD5_Sign(TST_ContextPrivate1,
												   TST_PrivateKey1,
												   pDataIn,
												   pSignOut,
												   &KeySize);
		break;
	default:
		break;
	}

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_RSA_PKCS1v15_xxx_Sign",
												FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End; 


	/*4. Preform the Verify operation */
	/**********************************/
	switch(hashMode)
	{
	case CRYS_HASH_SHA1_mode:
		TST_ATP_Error = CRYS_RSA_PKCS1v15_SHA1_Verify(TST_ContextPublic1,
													  TST_PublicKey1,
													  pDataIn,
													  pSignOut);   
		break;
	case CRYS_HASH_SHA224_mode:
		TST_ATP_Error = CRYS_RSA_PKCS1v15_SHA224_Verify(TST_ContextPublic1,
														TST_PublicKey1,
														pDataIn,
														pSignOut);    
		break;
	case CRYS_HASH_SHA256_mode:
		TST_ATP_Error = CRYS_RSA_PKCS1v15_SHA256_Verify(TST_ContextPublic1,
														TST_PublicKey1,
														pDataIn,
														pSignOut);     
		break;
	case CRYS_HASH_SHA384_mode:
#ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
		TST_ATP_Error = CRYS_RSA_PKCS1v15_SHA384_Verify(TST_ContextPublic1,
														TST_PublicKey1,
														pDataIn,
														pSignOut);     
		break;
#else
		goto End;
#endif
	case CRYS_HASH_SHA512_mode:
#ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
		TST_ATP_Error = CRYS_RSA_PKCS1v15_SHA512_Verify(TST_ContextPublic1,
														TST_PublicKey1,
														pDataIn,
														pSignOut);     
		break;
#else
		goto End;
#endif
	case CRYS_HASH_MD5_mode:  
		TST_ATP_Error = CRYS_RSA_PKCS1v15_MD5_Verify(TST_ContextPublic1,
													 TST_PublicKey1,
													 pDataIn,
													 pSignOut);     
		break;
	default:
		break;
	}     

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_RSA_PKCS1v15_xxx_Verify",
												FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;  

End:
	/* 5. Free allocated buffers */
	/*****************************/
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pSignOut);
	DX_VOS_MemFree(TST_PrivateKey1);
	DX_VOS_MemFree(TST_PublicKey1);
	DX_VOS_MemFree(TST_PrimeData1);
	DX_VOS_MemFree(TST_KeyGenData1);
	DX_VOS_MemFree(TST_ContextPrivate1);
	DX_VOS_MemFree(TST_ContextPublic1);

	return TST_TestStatus;	

}

/***************************
* Function Name:                                                           
*  SEPQA_RSA_PSS_ExternalHash_SignVerify         
* 
* Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* KeyGenType - key gen type (internal/external)
* CRT_Mode - mode of the private key
* hashMode - hash mode
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
*  dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
* FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for RSA PKCS1v2.1 PSS Sign & Verify
*  When the Hash for the Sign is preformed externally                   
*  This function Signs & Verifies hash vectors using RSA PKCS1v2.1 PSS                                                   
*  Key sizes used     (in bits) - 512/1024/2048
* Algorithm:       
*  1. Prepare keys 
*     a. Generate keys 
*     or
*     a. Build the Private Key from external exponent and modulus
*     b. Build the Public Key from External exponent and modulus 
*  2. Call the CRYS_RSA_PSS_SHA1_Sign or CRYS_RSA_PSS_MD5_Sign to preform the Sign operation
*  3. Call the CRYS_RSA_PSS_SHA1_Verify or CRYS_RSA_PSS_MD5_verify to complete the verification operation
*  4. Compare the outputted result to the expected one                  
*
**************************/
CRYSError_t SEPQA_RSA_PSS_ExternalHash_SignVerify(	DxUint8_t	          *PubExponent_ptr,
													DxUint16_t	           PubExponentSize,
													DxUint8_t             *PrivExponent_ptr,
													DxUint8_t	          *Modulus_ptr,
													DxUint16_t	           KeySize,
													DxUint8_t              CRT_Mode,
													DxUint8_t			   KeyGenType,
													CRYS_RSA_HASH_OpMode_t HashMode,
													DxUint16_t		       SaltLength,
													DxUint8_t             *P_ptr, 
													DxUint16_t	           PSize,
													DxUint8_t             *Q_ptr,
													DxUint16_t             QSize,
													DxUint8_t             *dP_ptr, 
													DxUint16_t             dPSize,
													DxUint8_t             *dQ_ptr,
													DxUint16_t             dQSize,
													DxUint8_t             *qInv_ptr,
													DxUint16_t             qInvSize,
													DxUint8_t             *Data_ptr,
													DxUint16_t             DataSize,
													DxUint8_t             *FailureFunctionNameOut)
{
	/*** TST Variables Definitions ****/
	CRYSError_t					TST_TestStatus = 0;
	
	DxUint8_t					*pDataIn = DX_NULL;
	DxUint8_t					*pSignOut = DX_NULL;
	CRYS_RSAUserPrivKey_t       *TST_PrivateKey1 = DX_NULL;
	CRYS_RSAUserPubKey_t        *TST_PublicKey1 = DX_NULL;
	CRYS_RSAPrimeData_t         *TST_PrimeData1 = DX_NULL;
	CRYS_RSAKGData_t            *TST_KeyGenData1 = DX_NULL;
	CRYS_RSAPrivUserContext_t   *TST_ContextPrivate1 = DX_NULL; 
	CRYS_RSAPubUserContext_t    *TST_ContextPublic1 = DX_NULL; 

	/* 1. Allocate memory buffers */
	/******************************/
	TST_ContextPrivate1 = DX_VOS_MemMalloc(sizeof(CRYS_RSAPrivUserContext_t));
	TST_ContextPublic1  = DX_VOS_MemMalloc(sizeof(CRYS_RSAPubUserContext_t));
	TST_PrivateKey1     = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPrivKey_t));
	TST_PublicKey1      = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPubKey_t));
	TST_KeyGenData1     = DX_VOS_MemMalloc(sizeof(CRYS_RSAKGData_t));
	TST_PrimeData1      = DX_VOS_MemMalloc(sizeof(CRYS_RSAPrimeData_t));
	pDataIn             = DX_VOS_MemMalloc(DataSize);
	pSignOut            = DX_VOS_MemMalloc(TST_MAX_PRVT_EXP_SIZE_IN_BYTES);

	if ((TST_ContextPrivate1 == DX_NULL) ||
		(TST_ContextPublic1 == DX_NULL)  ||
		(TST_PrivateKey1 == DX_NULL)     ||
		(TST_PublicKey1 == DX_NULL)      ||
		(TST_KeyGenData1 == DX_NULL)     ||
		(TST_PrimeData1 == DX_NULL)      ||
		(pDataIn == DX_NULL)             ||
		(pSignOut == DX_NULL))
	{
		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
													"DX_VOS_MemMalloc",
													FailureFunctionNameOut);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, Data_ptr, DataSize);

	/*Cleaning the keys to prevent test errors with KG*/ 
	DX_VOS_MemSet(TST_PrivateKey1,0,sizeof(CRYS_RSAUserPrivKey_t));
	DX_VOS_MemSet(TST_PublicKey1,0,sizeof(CRYS_RSAUserPubKey_t));
	DX_VOS_MemSet(TST_KeyGenData1,0,sizeof(CRYS_RSAKGData_t));
	DX_VOS_MemSet(TST_PrimeData1,0,sizeof(CRYS_RSAPrimeData_t));
	DX_VOS_MemSet(TST_ContextPrivate1,0,sizeof(CRYS_RSAPrivUserContext_t));
	DX_VOS_MemSet(TST_ContextPublic1,0,sizeof(CRYS_RSAPubUserContext_t));

	/* 2. Prepare keys */
	/*******************/
	if (KeyGenType == TST_INTERNAL_KG)
	{
		if (CRT_Mode == TST_NO_CRT)
		{
			/* Generate The Public and private keys */
			/**************/
			TST_ATP_Error = CRYS_RSA_KG_GenerateKeyPair(PubExponent_ptr,
														PubExponentSize,
														KeySize*8,
														TST_PrivateKey1,
														TST_PublicKey1,
														TST_KeyGenData1);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
														"CRYS_RSA_KG_GenerateKeyPair",
														FailureFunctionNameOut);
			if (TST_TestStatus != TESTPASS) 
				goto End;                                                               
		}
		else
		{
			/* Generate The Public and private CRT keys */
			/****************/
			TST_ATP_Error = CRYS_RSA_KG_GenerateKeyPairCRT(PubExponent_ptr,
														   PubExponentSize,
														   KeySize*8,
														   TST_PrivateKey1,
														   TST_PublicKey1,
														   TST_KeyGenData1);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
														"CRYS_RSA_KG_GenerateKeyPairCRT",
														FailureFunctionNameOut);
			if (TST_TestStatus != TESTPASS) 
				goto End;                                          
		}
	}
	else
	{                   
		/* Build the public Key */
		/*----------------------*/
		TST_ATP_Error = CRYS_RSA_Build_PubKey(TST_PublicKey1,
											  PubExponent_ptr,
											  PubExponentSize,
											  Modulus_ptr,
											  KeySize);


		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_RSA_Build_PubKey",
													FailureFunctionNameOut);
		if (TST_TestStatus != TESTPASS) 
			goto End;      

		/* Build the Private Key */
		/*-----------------------*/ 
		if (CRT_Mode == TST_NO_CRT)
		{
			TST_ATP_Error = CRYS_RSA_Build_PrivKey(TST_PrivateKey1,
												   PrivExponent_ptr,
												   KeySize,
												   PubExponent_ptr,
												   PubExponentSize,                                               
												   Modulus_ptr,
												   KeySize);                                  

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
														"CRYS_RSA_Build_PrivKey",
														FailureFunctionNameOut);
			if (TST_TestStatus != TESTPASS) 
				goto End;   
		}
		else
		{
			TST_ATP_Error = CRYS_RSA_Build_PrivKeyCRT(TST_PrivateKey1, P_ptr, PSize,
													  Q_ptr, QSize, dP_ptr, dPSize, 
													  dQ_ptr, dQSize, qInv_ptr, qInvSize);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
														"CRYS_RSA_Build_PrivKeyCRT",
														FailureFunctionNameOut);
			if (TST_TestStatus != TESTPASS) 
				goto End; 
		}
	}/*End of TST_KeyGenType*/

	/*3. Preform the Sign operation */
	/********************************/
	switch(HashMode)
	{
	case CRYS_HASH_SHA1_mode:
		TST_ATP_Error = CRYS_RSA_PSS_SHA1_Sign(TST_ContextPrivate1,
												TST_PrivateKey1,
												CRYS_PKCS1_MGF1,
												SaltLength,
												Data_ptr,
												pSignOut,
												&KeySize);
		break;
	case CRYS_HASH_SHA224_mode:
		TST_ATP_Error = CRYS_RSA_PSS_SHA224_Sign(TST_ContextPrivate1,
												 TST_PrivateKey1,
												 CRYS_PKCS1_MGF1,
												 SaltLength,
												 Data_ptr,
												 pSignOut,
												 &KeySize);
		break;
	case CRYS_HASH_SHA256_mode:
		TST_ATP_Error = CRYS_RSA_PSS_SHA256_Sign(TST_ContextPrivate1,
												 TST_PrivateKey1,
												 CRYS_PKCS1_MGF1,
												 SaltLength,
												 Data_ptr,
												 pSignOut,
												 &KeySize);
		break;
	case CRYS_HASH_SHA384_mode:
#ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
		TST_ATP_Error = CRYS_RSA_PSS_SHA384_Sign(TST_ContextPrivate1,
												 TST_PrivateKey1,
												 CRYS_PKCS1_MGF1,
												 SaltLength,
												 Data_ptr,
												 pSignOut,
												 &KeySize);
		break;
#else
		goto End;
#endif
	case CRYS_HASH_SHA512_mode:
#ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
		TST_ATP_Error = CRYS_RSA_PSS_SHA512_Sign(TST_ContextPrivate1,
												 TST_PrivateKey1,
												 CRYS_PKCS1_MGF1,
												 SaltLength,
												 Data_ptr,
												 pSignOut,
												 &KeySize);
		break;
#else
		goto End;
#endif
	default:
		break;
	}                                                                    
	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_RSA_PSS_XXX_Sign",
												FailureFunctionNameOut);
	if (TST_TestStatus != TESTPASS) 
		goto End; 



	/*4. Preform the Verify operation */
	/**********************************/
	switch(HashMode)
	{
	case CRYS_HASH_SHA1_mode:
		TST_ATP_Error = CRYS_RSA_PSS_SHA1_Verify(TST_ContextPublic1,
												 TST_PublicKey1,
												 CRYS_PKCS1_MGF1,
												 SaltLength,
												 Data_ptr,
												 pSignOut);
		break;
	case CRYS_HASH_SHA224_mode:
		TST_ATP_Error = CRYS_RSA_PSS_SHA224_Verify(TST_ContextPublic1,
												   TST_PublicKey1,
												   CRYS_PKCS1_MGF1,
												   SaltLength,
												   Data_ptr,
												   pSignOut);
		break;
	case CRYS_HASH_SHA256_mode:
		TST_ATP_Error = CRYS_RSA_PSS_SHA256_Verify(	TST_ContextPublic1,
													TST_PublicKey1,
													CRYS_PKCS1_MGF1,
													SaltLength,
													Data_ptr,
													pSignOut);
		break;
	case CRYS_HASH_SHA384_mode:
#ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
		TST_ATP_Error = CRYS_RSA_PSS_SHA384_Verify( TST_ContextPublic1,
													TST_PublicKey1,
													CRYS_PKCS1_MGF1,
													SaltLength,
													Data_ptr,
													pSignOut);
		break;
#else
		goto End;
#endif
	case CRYS_HASH_SHA512_mode:
#ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
		TST_ATP_Error = CRYS_RSA_PSS_SHA512_Verify( TST_ContextPublic1,
													TST_PublicKey1,
													CRYS_PKCS1_MGF1,
													SaltLength,
													Data_ptr,
													pSignOut);
		break;
#else
		goto End;
#endif
	default:
		break;
	}

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_RSA_PSS_XXX_Verify",
												FailureFunctionNameOut);
	if (TST_TestStatus != TESTPASS) 
		goto End; 

End:
	/* 5. Free allocated buffers */
	/*****************************/
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pSignOut);
	DX_VOS_MemFree(TST_PrivateKey1);
	DX_VOS_MemFree(TST_PublicKey1);
	DX_VOS_MemFree(TST_PrimeData1);
	DX_VOS_MemFree(TST_KeyGenData1);
	DX_VOS_MemFree(TST_ContextPrivate1);
	DX_VOS_MemFree(TST_ContextPublic1);

	return TST_TestStatus;	
}


/***************************
* Function Name:                                                           
*  SEPQA_RSA_Get_PrivAndPubKey      
* 
*Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* CRT_Mode - mode of the private key
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
* dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* 
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is an Acceptance test for RSA Get Privet Key and Get Public Key                  
*  This function Generate a RSA key and get its privet\public key and
*  compare it to the test vector
*
*  Key sizes used     (in bits) - 2048
*  This also outputs the results to an output buffer or the same input buffer
* Algorithm:       
*  1. Prepare keys 
*     a. Generate keys 
*     or
*     a. Build the Private Key from external exponent and modulus
*     b. Build the Public Key from External exponent and modulus
*  3. Call the CRYS_RSA_Get_Privet_key to preform the Get operation
*  4. Call the CRYS_RSA_Get_Public_key to preform the Get operation
*  5. Compare the outputted result to the expected one                  
*
**************************/
CRYSError_t SEPQA_RSA_Get_PrivAndPubKey(DxUint8_t	          *PubExponent_ptr,
										DxUint16_t	           PubExponentSize,
										DxUint8_t             *PrivExponent_ptr,
										DxUint16_t             PrivExponentSize,
										DxUint8_t	          *Modulus_ptr,
										DxUint16_t	           KeySize,
										DxUint8_t              CRT_Mode,
										DxUint8_t             *P_ptr, 
										DxUint16_t	           PSize,
										DxUint8_t             *Q_ptr,
										DxUint16_t             QSize,
										DxUint8_t             *dP_ptr, 
										DxUint16_t             dPSize,
										DxUint8_t             *dQ_ptr,
										DxUint16_t             dQSize,
										DxUint8_t             *qInv_ptr,
										DxUint16_t             qInvSize,
										DxUint8_t             *FailureFunctionNameOut)
{

	/*** TST Variables Definitions ****/
	CRYSError_t					TST_TestStatus = 0;
    SEP_RSA_TestGetKeyDataStruct TST_RSA_Tmp_Key;
	CRYS_RSAUserPrivKey_t       *TST_PrivateKey1 = DX_NULL;
	CRYS_RSAUserPubKey_t        *TST_PublicKey1 = DX_NULL;
	

	/* 1. Allocate memory buffers */
	/******************************/
	TST_PrivateKey1     = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPrivKey_t));
	TST_PublicKey1      = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPubKey_t));
	

	if ((TST_PrivateKey1 == DX_NULL) ||	(TST_PublicKey1 == DX_NULL))  
	
	{
		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
													"DX_VOS_MemMalloc",
													FailureFunctionNameOut);

		goto End;                                                                       
	}

	/*Cleaning the keys to prevent test errors with KG*/ 
	DX_VOS_MemSet(TST_PrivateKey1,0,sizeof(CRYS_RSAUserPrivKey_t));
	DX_VOS_MemSet(TST_PublicKey1,0,sizeof(CRYS_RSAUserPubKey_t));

	
	if (CRT_Mode == TST_NO_CRT)
	{
		/* 2. Generate The private key */
		/******************************/
		TST_ATP_Error = CRYS_RSA_Build_PrivKey( TST_PrivateKey1,
												PrivExponent_ptr, KeySize,
												PubExponent_ptr, PubExponentSize,                                               
												Modulus_ptr, KeySize);                                  

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_RSA_Build_PrivKey",
													FailureFunctionNameOut);
		if (TST_TestStatus != TESTPASS) 
			goto End; 

		/* 3. Build the public Key */
		/******************************/
		TST_ATP_Error = CRYS_RSA_Build_PubKey(TST_PublicKey1,
											  PubExponent_ptr, PubExponentSize,
											  Modulus_ptr, KeySize);


		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_RSA_Build_PubKey",
													FailureFunctionNameOut);
		if (TST_TestStatus != TESTPASS) 
			goto End; 

		TST_RSA_Tmp_Key.TST_PrivExponentSize = KeySize;
		TST_RSA_Tmp_Key.TST_PubExponentSize = PubExponentSize;
		TST_RSA_Tmp_Key.TST_ModulusSize = KeySize;

		TST_ATP_Error = CRYS_RSA_Get_PubKey(TST_PublicKey1,
											TST_RSA_Tmp_Key.TST_PublicExponent_E,
											&TST_RSA_Tmp_Key.TST_PubExponentSize,
											TST_RSA_Tmp_Key.TST_Modulus_N,
											&TST_RSA_Tmp_Key.TST_ModulusSize);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_RSA_Get_PubKey",
													FailureFunctionNameOut);
		if (TST_TestStatus != TESTPASS) 
			goto End; 


		/* 4. Comparing The Public Key - Exponent */
		/*****************************************/
		TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(TST_RSA_Tmp_Key.TST_PublicExponent_E,
												   PubExponent_ptr, PubExponentSize);


		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_RSA_Compare_Public_Key Exponent",
													FailureFunctionNameOut);
		if (TST_TestStatus != TESTPASS) 
			goto End;                                                             

		/* 5. Comparing The Public Key - Modulus */
		/*****************************************/
		TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(TST_RSA_Tmp_Key.TST_Modulus_N,
												   Modulus_ptr, KeySize);


		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_RSA_Compare_Public_Key Modulus",
													FailureFunctionNameOut);
		if (TST_TestStatus != TESTPASS) 
			goto End;  


		TST_ATP_Error = CRYS_RSA_Get_PrivKey(TST_PrivateKey1,
											 TST_RSA_Tmp_Key.TST_PrivetExponent_D,
											 &TST_RSA_Tmp_Key.TST_PrivExponentSize,
											 TST_RSA_Tmp_Key.TST_PublicExponent_E,
											 &TST_RSA_Tmp_Key.TST_PubExponentSize,
											 TST_RSA_Tmp_Key.TST_Modulus_N,
											 &TST_RSA_Tmp_Key.TST_ModulusSize);


		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_RSA_Get_PrivKey",
													FailureFunctionNameOut);
		if (TST_TestStatus != TESTPASS) 
			goto End; 

		/*6.Comparing The Private Key */
		/*****************************/
		TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(TST_RSA_Tmp_Key.TST_PrivetExponent_D,
												   PrivExponent_ptr, PrivExponentSize);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_RSA_Compare_Private_Key",
													FailureFunctionNameOut);
		if (TST_TestStatus != TESTPASS) 
			goto End;                                                            
	}
	else
	{
		/*1. Get The private CRT key */
		/****************************/
		TST_ATP_Error = CRYS_RSA_Build_PrivKeyCRT(TST_PrivateKey1,
												  P_ptr, PSize, Q_ptr, QSize,
												  dP_ptr,dPSize, dQ_ptr, dQSize,
												  qInv_ptr, qInvSize);	

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_RSA_Build_PrivKeyCRT",
													FailureFunctionNameOut);
		if (TST_TestStatus != TESTPASS) 
			goto End;   


		TST_RSA_Tmp_Key.TST_pSize    = 	PSize;
		TST_RSA_Tmp_Key.TST_qSize    = 	QSize;
		TST_RSA_Tmp_Key.TST_DpSize   = 	dPSize;
		TST_RSA_Tmp_Key.TST_DqSize   = 	dQSize;
		TST_RSA_Tmp_Key.TST_QinvSize = 	qInvSize;

		TST_ATP_Error = CRYS_RSA_Get_PrivKeyCRT(TST_PrivateKey1,
												TST_RSA_Tmp_Key.TST_P,
												&TST_RSA_Tmp_Key.TST_pSize,
												TST_RSA_Tmp_Key.TST_Q,
												&TST_RSA_Tmp_Key.TST_qSize,
												TST_RSA_Tmp_Key.TST_Dp,
												&TST_RSA_Tmp_Key.TST_DpSize,
												TST_RSA_Tmp_Key.TST_Dq,
												&TST_RSA_Tmp_Key.TST_DqSize,
												TST_RSA_Tmp_Key.TST_Qinv,
												&TST_RSA_Tmp_Key.TST_QinvSize);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_RSA_Get_PrivKeyCRT",
													FailureFunctionNameOut);
		if (TST_TestStatus != TESTPASS) 
			goto End;    

		/*2.Comparing The Private Key */
		/******************************/
		TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(TST_RSA_Tmp_Key.TST_P, P_ptr, TST_RSA_Tmp_Key.TST_pSize);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_RSA_Compare_CRT_Private_Key",
													FailureFunctionNameOut);
		if (TST_TestStatus != TESTPASS) 
			goto End;  
	}


End:
	/* Free allocated buffers */
	/*************************/
	DX_VOS_MemFree(TST_PrivateKey1);
	DX_VOS_MemFree(TST_PublicKey1);

	return TST_TestStatus;	
}

